// ---------------------------------------------------
// File: StackTrace.java
//
// Copyright 1998 Veo Systems
// Written by Kenneth Persson/Henrik Martin
// All Rights Reserved.
//
// $Id$
// ---------------------------------------------------

package com.amichel.util.debug;
 
import java.io.*;
import java.util.StringTokenizer;

/**
 * Class that implements a StackTrace and provided various nice
 * accessor methods for extracting things like the current line
 * number, the source file name, fully qualified class name, etc.
 * Based on some cool ideas originally provided by Kenneth Persson.
 * This class is most likely to be used by other convenience classes
 * like the com.commerceone.util.debug.Debug class.
 *
 * @see java.lang.Throwable
 * @version $Revision: 1.10 $
 */

public class StackTrace
{
  private int stackLevels = 1; // # of stack trace lines to print (default 1)
  private int skipDepth = 0; // # of stack trace lines to skip
  private int savedLines = 0;
  private StringWriter sw = null; // Temp container for stack trace
  private StringTokenizer strTok = null;
  private Throwable t = null; // The object containing the stack trace
  private String fileName = null, myName = getClass().getName(),
    tmpStr = null, className = null, kitchenSink = null,
    methodName = null, fullClassName = null;
  private StringBuffer stackTrace = null;
  private int lineNum = -1; // -1 in case tokenizing yields an exception

  // The throwable that creates the stack trace printout is
  // created in the StackTrace() constructor.
  // This will cause the top of the stack to be StackTrace.<init>, which
  // has to be filtered out from the stack trace we return. Since the
  // no-arg StackTrace() constructor calls the other constructor, this
  // will cause yet another line from this class to be added to the stack
  // trace. Basically any method call done in this class will be added
  // to the stack trace, but we don't want to return that to the caller.
  // 
  // So once the Throwable object is constructed, its printStackTrace()
  // method is going to be called like this:
  // t.printStackTrace(new PrintWriter(sw));
  // The 'sw' parameter is an object of type StringWriter. What this
  // enables us to do is to obtain a string containing the entire stack
  // trace by invoking sw.toString(). This very cool idea was provided
  // by Kenneth Persson.
  // The only remaining thing is to filter out any stack lines added by
  // this class. So we tokenize the String and skip those lines that
  // match our class name.
  public StackTrace()
  {
    this(1, 0);
  }
  public StackTrace(int stackLevels)
  {
    this(stackLevels, 0);
  }
  public StackTrace(int stackLevels, int skipDepth)
  {
    // Make sure we get sane values in
    if (Contract.REQUIRE)
    {
      Contract.require(stackLevels > 0);
      Contract.require(skipDepth >= 0);
    }
    
    this.stackLevels = stackLevels; // # of lines of stack trace to save
    this.skipDepth = skipDepth; // # of lines to skip from top of stack trace

    sw = new StringWriter(); // Container that we pass to the PrintWriter

    // Produce the stack trace and save it by printing to our dummy sw object
    t = new Throwable();
    t.printStackTrace(new PrintWriter(sw));

    // Tokenize the stack trace and parse it to filter out unwanted lines
    stackTrace = new StringBuffer();
    strTok = new StringTokenizer(sw.toString(), "\n\r");

    try
    {
      // First line is always "java.lang.Throwable", so just skip it
      if (strTok.hasMoreTokens())
      {
	strTok.nextToken();
      }
      
      // Skip over all lines matching our class name, starting from the top
      while (strTok.hasMoreTokens())
      {
	// Jview use '/' as class component sep
	tmpStr = strTok.nextToken().replace('/', '.');
	if (tmpStr.indexOf(myName, 0) != -1)
	{
	  // Matched our class name, just skip over it
	  continue;
	}

	// Possibly skip a number of lines in the stack trace.
	if (skipDepth > 0)
	{
	  skipDepth--;
	  continue;
	}
	else
	{
	  break;
	}
      }

      // Second time around, now we want to save the remaining tokens
      // up to the number of lines that the caller requested.
      while (savedLines < stackLevels)
      {
	stackTrace.append(tmpStr.trim());
	stackTrace.append("\n");
	savedLines++;

	if (! strTok.hasMoreTokens())
	{
	  break;
	}
	tmpStr = strTok.nextToken();
      }

      // Extract miscellaneous information from the stack trace.
      // The way this is done is by tokenizing the stack trace
      // on parenthesis, whitespace and colons and save the various
      // tokens.
      // This is an ugly solution, based on assumptions
      // about stack trace format (this may change in the future), but
      // what's a poor programmer to do when he needs the feature???
      strTok = new StringTokenizer(stackTrace.toString(), " ():");

      // Waste the first token ("at")
      strTok.nextToken();

      // Get whatever is between the parenthesis File:num
      kitchenSink = strTok.nextToken(); // Read past first token...
      fileName = strTok.nextToken(); // ...save the file name...
      tmpStr = strTok.nextToken(); // ...here's our line # string
      lineNum = Integer.parseInt(tmpStr); // Now convert String to int

      // Extract the class name by changing .java to .class in the filename
      className = fileName.substring(0, fileName.indexOf(".java")) + ".class";
      // Extract the full package name
      methodName = kitchenSink.substring(kitchenSink.lastIndexOf(".") + 1);
      fullClassName =
	kitchenSink.substring(0, kitchenSink.lastIndexOf(".")) + ".class";
    }
    catch (Exception e) 
    { 
	// This will happen e.g. if the code is compiled without
	// Debug info. Don't print anything. The Debug class now
	// takes care of the problem.

	// System.err.println("Exception: " + e.getMessage()); 
    }
  } // public StackTrace(int stackLevels, int skipDepth)

  /**
   * Method that returns entire stack trace as a string
   */
  public String toString() {
    return stackTrace.toString();
  }

  /**
   * Method that returns the current line # for the class created
   * the StackTrace object. Returns -1 if the current line # can't
   * be determined.
   */
  public int getLineNum() {
    return lineNum;
  }

  /**
   * Method that returns the source file's name for the class
   * that created the StackTrace object. Returns null if the
   * file name can't be determined.
   */
  public String getFileName() {
    return fileName;
  }

  /**
   * Method that combined getFileName() and getLineNum(). Returns
   * a string of the type "Foo.java:10" or "null:-1" if the file name
   * and line number can't be determined.
   */
  public String getFileAndLineNum()
  {
    return fileName + ":" + lineNum;
  }

  /**
   * Returns the fully qualified package name for the class
   * that created the StackTrace object or null if that can't
   * be determined.
   */
  public String getFullClassName() {
    return fullClassName;
  }

  /**
   * Returns the class name for the class that created the
   * StackTrace object or null if that can't be determined.
   */
  public String getClassName() {
    return className;
  }

  /**
   * Returns the name of the currently executing method from
   * the class that created the StackTrace object or null
   * if that can't be determined.
   */
  public String getMethodName() {
    return methodName;
  }

  protected static boolean isDebugInfoAvailable()
  {
    StackTrace ln = new StackTrace();
    return ln.getLineNum() != -1;
  }

} // public class StackTrace
