// ---------------------------------------------------
// File: RegexFilterWriter.java
//
// Copyright (C) 1998 Veo Systems, Inc.
// Written by Henrik Martin
// All rights reserved.
//
// $Id: RegexFilterWriter.java,v 1.10 1999/03/10 06:53:00 kenneth Exp $
// ---------------------------------------------------

package com.amichel.util.debug;

import java.io.*;
import com.amichel.util.regex.RegExp;
import com.amichel.util.regex.RegExpSyntaxException;
import com.amichel.util.regex.NFABuildException;

  
/**
 * Class that makes up a <code>RegexFilterWriter</code>, used to filter
 * characters sent to a stream (most likely
 * <code>System.err/System.out</code>). Only strings that match the
 * currently installed regular expression will be forwarded to the next
 * output stream in the chain. Here's an example of how this can be used:
 * <p><blockquote><code><pre>
 * FileOutputStream fos = new FileOutputStream("log");
 * PrintWriter pw = new PrintWriter(new RegexFilterWriter(fos, "DEBUG"), true);
 * pw.println("This line won't show up in the file");
 * pw.println("But this one will since it contains the string \"DEBUG\"");
 * </blockquote></code></pre>
 * Or if you need to filter out some annoying debug messages that some
 * class out of your control logs to <code>System.out/System.err</code>:
 * <p><blockquote><code><pre>
 * // Create System.err filter that only lets messages matching SOMEREGEX pass
 * RegexFilterWriter rfw = new RegexFilterWriter(System.err, "SOMEREGEX");
 * // Wrap its OutputStream in a PrintStream
 * PrintStream ps = new PrintStream(rfw.getOutputStream(), true);
 * // Tell System.err to log to the filter stream instead of the TTY.
 * System.setErr(ps);
 * </blockquote></code></pre>
 * After the code above is executed, everything that doesn't match the
 * regular expression SOMEREGEX will be silently filtered and never
 * printed to <code>System.err</code>.
 *
 * @see java.io.FilterWriter
 * @see java.io.PrintStream
 * @see java.io.PrintWriter 
 * @see regex.RegExp
 * @see regex.MatchInfo
 * @see com.commerceone.util.debug.Debug
 * @see com.commerceone.util.contract.Contract
 * @version $Revision: 1.10 $
 */
public class RegexFilterWriter extends FilterWriter
{
  private PrintWriter pw = null;
  private RegExp regExp = null;
  private OutputStream proxyStream = null;
  private boolean didFilter = false;

  /**
   * Constructs an instance of RegexFilterWriter attached
   * to System.err with the regular expression set to
   * <code>matchString.</code>
   * @param String matchString.
   *
   * @exception regex.RegExpSyntaxException
   * @exception regex.NFABuildException
   */
  public RegexFilterWriter(String matchString)
    throws com.amichel.util.regex.RegExpSyntaxException, com.amichel.util.regex.NFABuildException
  {
    // If caller didn't specify otherwise, install stream on System.err
    this(System.err, matchString);
  } 

  /**
   * Constructs an instance of RegexFilterWriter attached
   * to the PrintWriter stream <code>pw</code>, filtering
   * output using the regular expression specified by 
   * <code>matchString.</code>
   * @param java.io.PrintWriter pw
   * @param String matchString.
   *
   * @exception regex.RegExpSyntaxException
   * @exception regex.NFABuildException
   */
  public RegexFilterWriter(PrintWriter pw, String matchString)
    throws RegExpSyntaxException, NFABuildException
  {
    super(pw); // Have to create dummy stream 
    
    // Parameter sanity checking
    if (Contract.REQUIRE)
    {
      Contract.require(pw != null);
      Contract.require(matchString != null);
    }
    
    this.pw = pw;

    // This line is VERY important. Do NOT remove it.
    super.out = this.pw;
    
    setRegExp(matchString);    
  } // public RegexFilterWriter(PrintWriter newPw, String matchString)

  /**
   * Constructs an instance of RegexFilterWriter attached
   * to the OutputStream <code>os</code>, filtering
   * output using the regular expression specified by 
   * <code>matchString.</code>
   * @param java.io.OutputStream os
   * @param String matchString.
   *
   * @exception regex.RegExpSyntaxException
   * @exception regex.NFABuildException
   */
  public RegexFilterWriter(OutputStream os, String matchString)
    throws RegExpSyntaxException, NFABuildException
  {
    super(new PrintWriter(os));

    // Parameter sanity checking
    if (Contract.REQUIRE)
    {
      Contract.require(os != null);
      Contract.require(matchString != null);
    }
    
    // If the supplied stream is System.err, we must crawl in under
    // it and grab the TTY handle. Otherwise, if someone passes our
    // stream to System.setErr()/setOut(), we'll lose the TTY.
    // The way we ensure a handle to the TTY is by utilizing
    // FileDescriptor.err or FileDescriptor.out.
    if (os.equals(System.err))
    {
      pw = new PrintWriter(new FileOutputStream(FileDescriptor.err), true);
    }
    else if (os.equals(System.out))
    {
      pw = new PrintWriter(new FileOutputStream(FileDescriptor.out), true);
    }
    else
    {
      // Wrap the PrintStream in a PrintWriter
      pw = new PrintWriter(os, true);
    }

    this.pw = pw;
    
    // This line is VERY important. Do NOT remove it.
    super.out = this.pw;

    setRegExp(matchString);
  } // public RegexFilterWriter(OutputStream newPs, String matchString)  


  // Various println() methods
  public synchronized void println(String str)
  {
    if (! filter(str))
    {
      pw.println(str);
    }
  }
  public synchronized void println(Object o)
  {
    println((o == null ? "null" : o.toString()));
  }
  public synchronized void println(char c)
  {
    pw.print(c);
  }
  public synchronized void println(char[] ca)
  {
    println(new String(ca));
  }

  // Various print() methods
  public synchronized void print(String str)
  {
    if (! filter(str))
    {
      pw.print(str);
    }
  }
  public synchronized void print(Object o)
  {
    print((o == null ? "null" : o.toString()));
  }
  public synchronized void print(char c)
  {
    pw.print(c);
  }
  public synchronized void print(char[] ca)
  {
    print(new String(ca));
  }

  // Various write() methods
  public synchronized void write(String str)
  {
    write(str, 0, str.length());
  }
  public synchronized void write(String str, int off, int len)
  {
    // If the length of the String is 1, then there's no need
    // to even try to filter it. This will be the case when
    // The wrapping FilterWriter flushes the output stream on
    // a println() call. So if the previous write was something
    // that passed the filter, then print out the newline, else
    // ignore it.
    if (len <= 1)
    {
      if (didFilter)
      {
	pw.write(str);
	didFilter = false;
      }
    }
    
    if (! filter(str))
    {
      pw.write(str, off, len);
      didFilter = true;
    }
  }
  public synchronized void write(char[] ca)
  {
    write(new String(ca));
  }
  public synchronized void write(char[] ca, int off, int len)
  {
    write(new String(ca, off, len));
  }
  public synchronized void write(byte[] ba)
  {
    write(new String(ba, 0, ba.length));
  }
  public synchronized void write(byte[] ba, int off, int len)
  {
    write(new String(ba, off, len));
  }
  
  // Method that returns true or false depending if the string
  // passed to it matches the installed RegExp.
  private synchronized boolean filter(String str)
  {
    // Make sure that the caller has installed a useful RegExp
    if (Contract.REQUIRE) { Contract.require(regExp != null); }
  
    return (regExp.match(str) == null ? true : false);
  } // public boolean filter(String str)


  /**
   * Method that installs a new regular expression pattern
   *
   * @param String newPat
   * @exception regex.RegExpSyntaxException
   * @exception regex.NFABuildException
   */
  public synchronized void setRegExp(String newPat)
    throws RegExpSyntaxException, NFABuildException
  {
    // Check parameter sanity
    if (Contract.REQUIRE) { Contract.require(newPat != null); }
    
    if (regExp == null)
    {
      regExp = new RegExp(newPat);
    }
    else
    {
      regExp.setPattern(newPat);
    }
  } // public void setRegExp(String newPat)


  /**
   * Method that returns this stream as an OutputStream. It
   * does this by creating an anonymous adapter class that
   * acts as a proxy, forwarding all calls to the write()
   * methods of the OutputStream object to the appropriate
   * methods in this class. The methods need not be synchronized
   * since the calling stream will synchronize its methods.
   */
  public OutputStream getOutputStream()
  {
    if (proxyStream == null)
    {
      proxyStream = new BufferedOutputStream(new ProxyOutputStream());
    }
    return proxyStream;
  }

  /**
   * Private member style inner class that acts as a OutputStream
   * adapter or proxy class so that this class that derives from
   * Writer, can be used as an OutputStream. This is needed if the
   * stream is going to be used on top of a socket, etc.
   */ 
  private class ProxyOutputStream extends OutputStream
  {
    private RegexFilterWriter rfw = RegexFilterWriter.this;

    public void close() throws IOException { rfw.close(); }
    public void flush() throws IOException { rfw.flush(); }
    public void write(int b) throws IOException { rfw.write(b); }
    public void write(byte[] ba) throws IOException
    {
      rfw.write(new String(ba, 0, ba.length));
    }
    public void write(byte[] ba, int off, int len)
      throws IOException
    {
      // Is there a more efficient way to do this?
      rfw.write(new String(ba, off, len));
    }
  } // private class ProxyOutputStream extends OutputStream

    
  // main method. For debugging and test purposes only
/*  public static void main(String[] args)
  {
    Test t = new Test();
    t.run();
  } // public static void main(String[] args)
*/
  private static class Test
  {
    RegexFilterWriter rfw = null;
    
    public void run()
    {
      System.err.println("*** Whitebox test for class RegexFilterWriter "
			 + "starting ***");
      System.err.println("*** Constructing instance of RegexFilterWriter "
			 + "with regex filter \"kalle\" ***"); 
      try { rfw = new RegexFilterWriter(System.err, "kalle"); }
      catch (Exception e)
      {
	System.err.println("Oops! Something crapped out: "
			   + e.toString() + "\n*** Bailing out ***");
	return;
      }
      System.err.println("*** Trying to write \"kalle\" to stream. "
			 + "You should see it show up below ***\n");
      rfw.println("kalle");
      System.err.println("\n*** If nothing showed up, that's "
			 + "an error ***");
      System.err.println("*** Trying to write \"nisse\" to stream. "
			 + "It should NOT show up on the screen! ***");
      rfw.println("nisse");      
      System.err.println("\n*** I hope nothing showed up ***");
    } // public void run()
  }// private static class Test
} // public class RegexFilterWriter extends OutputStream



