// ---------------------------------------------------
// File: Debug.java
//
// Copyright 1998 Veo Systems
// Written by Kenneth Persson/Henrik Martin
// All Rights Reserved.
//
// $Revision: 1.23 $ 
// $Date: 1999/03/10 06:53:00 $
// $Author: Henrik/kenneth $
//
// $Id$
// ---------------------------------------------------

package com.amichel.util.debug;
 
import java.io.*;
import java.util.Date;
import java.text.DateFormat;
import java.util.TimeZone;

/**
 * Class that provides a simple, generic interface to error
 * logging. It tags output with a log level, source filename/linenumber,
 * etc. In it's simplest form, this is how you would use it:
 * <pre><blockquote><code>
 * Debug log = new Debug();
 * log.debug("This is a message");
 * </pre></blockquote></code>
 * Which would cause a string like this to be printed to System.err:
 * <pre>
 * Debug.DEBUG: Foo.java:10: This is a message
 * </pre>The example shows a message generated from line 25 in the file
 * Foo.java. The default behavior for the <b>Debug</b> class is to
 * log messages to <b>System.err</b>, using a prefix of
 * "Debug.facility" where <b>facility</b> is one of <b>DEBUG, WARNING,
 * ERROR, INFO, CRITICAL, or FATAL</b> (depending on which of the debug(), warn(),
 * error(), crit(), info(), or fatal() methods was invoked for printing the message).
 * You can tailor how you want the output to look by explicitly calling the
 * method setLogParams() and pass it the prefix to use, and a boolean that
 * tells the class whether to use the filename/linenumber feature.<p>
 * Here's an example of how to redirect all your log output to the file
 * "log.out":
 * <pre><blockquote><code>
 * FileOutputStream fos = new FileOutputStream("log.out"); // Create log file
 * Debug log = Debug.getDefaultInstance();
 * log.setLogStream(fos); // Tell the Debug object to log to the file
 * log.debug("Microsoft sucks");
 *</pre></blockquote></code>
 * Finally, here's an example of how to combine the <b>Debug</b> class with
 * the <b>RegexFilterWriter</b> class to filter out all log messages except
 * those that contain the string "ERROR:":
 * <pre><blockquote><code>
 * RegexFilterWriter rfw = new RegexFilterWriter("ERROR:");
 * Debug log = Debug.getDefaultInstance();
 * log.setLogWriter(new PrintWriter(rfw, true));
 * log.debug("This message will never show up");
 * log.error("But this will, since it contains the string \"ERROR:\"");
 *</pre></blockquote></code>
 * Note, that the filtering takes place on the <b>entire</b> string, i.e. 
 * both the prefix and the message.
 *
 * @see com.commerceone.util.debug.Logger
 * @see com.commerceone.util.debug.RegexFilterWriter
 *
 * @version $Revision: 1.23 $
 */
public class Debug implements Logger
{
  private boolean useFileAndNum; 
  private boolean loggingEnabled = true;
  private PrintWriter logWriter = null; // Default stream for logging
  private OutputStream logStream = null; // Original log stream
  private String logPrefix = DEFAULT_LABEL; // Log message prefix
  private int logLevel = LOWEST; // Set the logLevel by default so that all messages get through

  // If compiled without the debug flag or running on a non
  // compliant JVM the value will be false, else true
  public static final boolean FILE_AND_LINE_AVAILABLE = 
      StackTrace.isDebugInfoAvailable();

  // "Shortcut" static copy for lazy hackers. This way you don't have
  // to instantiate the Debug class if you just want to do simple logging
  // to the default log stream (System.err).
  public static Debug log;

  static
  {
    log = new Debug();
  }

  public static void initDebug()
  {
    log = new Debug();
  }

  /**
   * Method that ensures that the Debug class is a Singleton. This is
   * important if someone wants to turn on or off debugging for an
   * entire application.
   */
  public static synchronized Debug getDefaultInstance()
  {
    return log; // Always instantiated by static initializer
  } // public static synchronized Debug getDefaultInstance()

  // Constructors
  protected Debug()
  {
    this(System.err, DEFAULT_LABEL, true);
  }

  protected Debug(RegexFilterWriter rfw)
  {
    this(rfw, DEFAULT_LABEL, true);
  }

  protected Debug(RegexFilterWriter rfw, String prefix, boolean useFileAndNum)
  {
    super();

    logStream = rfw.getOutputStream();
    logPrefix = prefix;

    this.setUseFileAndNum(true);
    this.setLogWriter(new PrintWriter(rfw, true));
  }

  protected Debug(OutputStream os)
  {
    this(os, DEFAULT_LABEL, true);
  } 
   
  protected Debug(OutputStream os, String prefix, boolean useFileAndNum)
  {
    super();

    logStream = os;
    logPrefix = prefix;

    this.setUseFileAndNum(useFileAndNum);
    this.setLogWriter(new PrintWriter(os, true));
  }

  /**
   * Methods for printing a log message to a log receiver.
   * The log receiver could be a PrintStream such as System.err,
   * or an OutputStream connected either to a file or a Syslog
   * client that forwards the log message to Syslogd (executing
   * either locally or on a remote host).
   * The stream is either the one that was passed to one of the
   * constructors for this class (or System.err if the no-arg
   * constructor was used), or one explicitly installed
   * by a setLogStream() call.
   *
   * @param java.lang.String msg
   */
  public void debug(String msg)
  {
    logIt(DEBUG, useFileAndNum, msg);
  }
  public void debug(Object obj)
  {
    debug(obj.toString());
  }
  
  public void warn(String msg)
  {
    logIt(WARNING, useFileAndNum, msg);
  }
  public void warn(Object obj)
  {
    warn(obj.toString());
  }

  public void error(String msg)
  {
    logIt(ERROR, useFileAndNum, msg);
  }
  public void error(Object obj)
  {
    error(obj.toString());
  }

  public void info(String msg)
  {
    logIt(INFO, useFileAndNum, msg);
  }
  public void info(Object obj)
  {
    info(obj.toString());
  }

  public void crit(String msg)
  {
    logIt(CRITICAL, useFileAndNum, msg);
  }
  public void crit(Object obj)
  {
    crit(obj.toString());
  }
  
  public void fatal(String msg)
  {
    logIt(FATAL, useFileAndNum, msg);
  }
  public void fatal(Object obj)
  {
    fatal(obj.toString());
  }

  private synchronized void logIt(int level,
				  boolean useFileAndNum, String msg)
  {
    // Only print out stuff if logging is turned on
	// and the current logging level is not greater than the level of the message
    if (! loggingEnabled || logLevel > level)
    {
      return;
    }
	DateFormat df = DateFormat.getDateTimeInstance(DateFormat.LONG,
                                                   DateFormat.LONG);

        
    //Use UTC time for time output
    TimeZone tz = TimeZone.getDefault();
    tz.setID("UTC");
    df.setTimeZone(tz);                 
    
	logWriter.println(df.format(new Date()) + ": " + 
					  (useFileAndNum ?
		       new StackTrace(1, 2).getFileAndLineNum() + ": " : "")
		      + logPrefix + "." + dbgNames[level] + ": " + msg);
  }

  /**
   * Reset the level to the default which is LOWEST
   * @return Returns the previous level
   */
  public int resetLevel() 
  { 
	int oldlevel = logLevel;
  	logLevel = LOWEST; 
	return oldlevel;
  }

  /**
   * Set the current logging level.
   * if the level is greater than LOWEST, 
   * then messages with a level lower than the current level are not output
   * @param iLevel The value of the new level
   * @return Returns the previous level
   */
  public synchronized int setLevel(int iLevel) 
  { 
	int oldlevel = logLevel;
  	logLevel = iLevel; 
	return oldlevel;
  }
  
  /**
  *	Return the current logging level.
  *	If the level is greater than LOWEST, 
  * then messages with a level lower than the current level are not output
  */
  public int getLevel() { return logLevel; }

  /**
   * Tests if logging is enabled or disabled. Implemented per Alex's
   * RFE. He wants to be able to log conditionally and to test for
   * the current state of the setting.
   */
  public boolean isEnabled() { return loggingEnabled; }

  /**
   * Enables or disables logging. Implemented per Alex's
   * RFE. He wants to be able to control whether or not to log
   * messages.
   */
  public synchronized void setEnabled(boolean state)
  {
    loggingEnabled = state;
  }

  /**
   * Method that changes the output format by letting
   * the caller change logging prefix and/or filename/linenumber
   * appended. 
   *
   * @param String newPrefix
   * @param boolean useFileAndNum
   */
  public synchronized void setLogParams(String newPrefix,
					boolean useFileAndNum)
  {
    // Check parameter sanity
    if (Contract.REQUIRE) { Contract.require(newPrefix != null); }

    // If newPrefix is A null string, it means that we shouldn't change
    // the log prefix
    if (newPrefix.compareTo("") != 0)
    {
      logPrefix = newPrefix;
    }
    
    this.setUseFileAndNum(useFileAndNum);
  } // public void setLogParams(String newPrefix, boolean useFileAndNum)

    /**
     * If not compiled with debug information or running with
     * jit compiler there is no idea to try to get line numbers.
     * This will only be set if information can be retreived.
     */
    private synchronized void setUseFileAndNum(boolean useFileAndNum)
    {
	if (Debug.FILE_AND_LINE_AVAILABLE)
	{
	    this.useFileAndNum = useFileAndNum;
	}
    }

  /**
   * Accessor method to retrieve the current log prefix
   */
  public String getPrefix() { return logPrefix; }

  /**
   * Method that changes the stream which logging messages are sent to.
   *
   * @param java.io.OutputStream os
   */
  public synchronized void setLogStream(OutputStream os)
  {
    // Check parameter sanity
    if (Contract.REQUIRE) { Contract.require(os != null); }

    logStream = os;
    logWriter = new PrintWriter(os, true);
  }
  /**
   * Same as above, but with a
   * @param java.io.PrintWriter as the stream which to print to.
   * PrintWriter is the preferred stream type, if not using
   * System.{out,err}, which unfortunately use the obsolete
   * PrintStream type.
   */
  public synchronized void setLogWriter(Writer w)
  {
    // Check parameter sanity
    if (Contract.REQUIRE) { Contract.require(w != null); }
    
    // Wrapping the PrintWriter in yet another PrintWriter is necessary
    // to enforce autoflushing of the stream.
    logWriter = new PrintWriter(w, true);
  }

  /**
   * Accessor method that returns the currently installed log stream.
   */
  public PrintWriter getLogWriter()
  {
    return logWriter;
  }

  public OutputStream getLogStream()
  {
    return logStream;
  }

  public synchronized void printStackTrace(Throwable t)
  {
    t.printStackTrace(logWriter);
  }
  
  public synchronized void printStackTrace()
  {
    logWriter.println(new StackTrace(200));
  }
} // public class Debug
