
package com.amichel.util.debug;

/**
 * Contract is a class, which encapsulates the programming
 * by contract ideas, expressed primarily by Bertrand
 * Meyer and the Eiffel community. C and C++ programs
 * often use macros based on <assert.h> which is controled
 * by compilation flags in Make.
 * <p>
 * Every pre- or post condition you add to your system
 * makes it better in the long run. Sometimes you will
 * express a contract, which later proves to be too
 * strict, and you relax it. Sometimes you will even
 * discover a contract that need to be a normal runtime
 * test instead. It is important to make a strict distinction
 * between programming contracts (programming errors) from
 * normal runtime exceptions.
 * <p>
 * An implementor of a method need to specify the exact
 * requirements for how to use it. Without it will often
 * be the case that the contract is underspecified, and
 * neither or both caller and implementor check some
 * variants.
 * <p>
 * To test that pre conditions are not invalidated, the
 * implementor of a method need to put this first in a method. 
 * <p>
 * <pre>
 * if (Contract.REQUIRE) Contract.require('expr');
 * </pre>
 * <p>
 * Where <code>'expr'</code> can be an arbitrary complex
 * boolean expression.
 * <p>
 * The test is not absolutely necissary, but will save a
 * method invocation if the the REQUIRE contract is turned
 * off (e.g. for deployment). More so, a smart compiler
 * will be able to not generate code for it at all.
 * <p>
 * If the expression is validated to be false, the program
 * is stopped, to give the programmer a chance to see, in
 * what way he missunderstood the contract for the method
 * invocation. In server implementations, and e.g. web server
 * collocation, you might not want to allow the program to
 * just exit. Take precausion to handle these situations, but
 * don't extrapolate this fact to become: programming by
 * contract is impossible for server implementations.
 * <p>
 * <code><pre>
 * Contract$ProgrammerError: REQUIRE
 *       at Contract.require(Contract.java:157)
 *       at TestContract.main(TestContract.java:32)
 * </pre></code>
 * <p>
 * The programmer who wrote the main function in class
 * TestContract, broke a contract. Go look around line 32
 * to find a clue.
 * <p>
 * While REQUIRE is the contract the implementor dictates
 * for callers to live up to. In return, the implementor
 * promises to live up to the contract specified in ENSURE.
 * <p>
 * <pre>
 * if (Contract.ENSURE) Contract.ensure('expr');
 * </pre>
 * <p>
 * If possible put the ensure in a finally block, so that both
 * normal cases and exceptional cases maintain the class
 * invariant.
 * <p>
 * There is also a generic CHECK, which can be used in code
 * not part of the pre and post condition contract. Can be
 * used to validate e.g. class invariants.
 * <p>
 * <pre>
 * if (Contract.CHECK) Contract.check('expr');
 * </pre>
 * <p>
 */

public class Contract 
{
    // ---------------------------------------------------
    // Static Constants

    /**
     * REQUIRE is always true during development. When the
     * build master is confident enough that no contract
     * violations are left in the code, he might choose to
     * set REQUIRE to false. Of the three constants REQUIRE,
     * ENSURE and CHECK, REQUIRE is the most important, and
     * therefore you'll keep it true for most of the time.
     * Let's put it this way: you don't want your code to 
     * break because somebody is violating your contract.
     * <p>
     * Rarely true in a deployed system, could be true in an
     * alpha or beta.
     *
     * @see com.commerceone.vsp.Contract#require()
     */
    public static final boolean REQUIRE=true;

    /**
     * ENSURE is mostly true during devlopment. When the build
     * build master is confident enough that implementors of
     * their methods have return in a consistent state and not
     * violating the post conditions. This is much easier to
     * predict and evaluate. Never true in a deployed system.
     *
     * @see com.commerceone.vsp.Contract#ensure()  
     */
    public static final boolean ENSURE=true;

    /**
     * CHECK can be true or false during devlopment. Mostly
     * true. Never true in a deployed system.
     *
     * @see com.commerceone.vsp.Contract#check()
     */
    public static final boolean CHECK=true;

        
    // ---------------------------------------------------
    // Static Methods

    /**
     * Require raises a private runtime exception if the
     * expression expr is evaluated to false. The exception
     * is meant to halt the execution and give a stack trace
     * where the method invocation is shown.
     * <p>
     * To test that pre conditions are not invalidated, the
     * method implementor needs to put this first in a method. 
     * <p>
     * <pre>
     * if (Contract.REQUIRE) Contract.require('expr');
     * </pre>
     *
     * @param     expr  a boolean expression to be evaluated
     * @exception com.commerceone.vsp.Contract#ProgrammerError
     * @see       com.commerceone.vsp.Contract#REQUIRE
     */
    public static void require(boolean expr)
        throws ProgrammerError
    {
        if (!expr) throw new ProgrammerError("REQUIRE");
    }

    public static void require(boolean expr, String msg)
        throws ProgrammerError
    {
        if (!expr) throw new ProgrammerError("REQUIRE: " + msg);
    }

    /**
     * Ensure raises a private runtime exception if the
     * expression expr is evaluated to false. The exception
     * is meant to halt the execution and give a stack trace
     * where the method it called from is shown.
     * <p>
     * To test that the post condition is not invalidated, the
     * programmer need to put this last in a method. 
     * <p>
     * <pre>
     * if (Contract.ENSURE) Contract.ensure('expr');
     * </pre>
     *
     * @param     expr  a boolean expression to be evaluated
     * @exception com.commerceone.vsp.Contract#ProgrammerError
     * @see       com.commerceone.vsp.Contract#ENSURE
     */
    public static void ensure(boolean expr)
        throws ProgrammerError
    {
        if (!expr) throw new ProgrammerError("ENSURE");
    }

    public static void ensure(boolean expr, String msg)
        throws ProgrammerError
    {
        if (!expr) throw new ProgrammerError("ENSURE:" + msg);
    }

    /**
     * Check raises a private runtime exception if the
     * expression expr is evaluated to false. The exception
     * is meant to halt the execution and give a stack trace
     * where the method invocation is shown.
     * <p>
     * <pre>
     * if (Contract.CHECK) Contract.check('expr');
     * </pre>
     *
     * @param     expr  a boolean expression to be evaluated
     * @exception com.commerceone.vsp.Contract#ProgrammerError
     * @see       com.commerceone.vsp.Contract#CHECK
     */
    public static void check(boolean expr)
        throws ProgrammerError
    {
        if (!expr) throw new ProgrammerError("CHECK");
    }

    public static void check(boolean expr, String msg)
        throws ProgrammerError
    {
        if (!expr) throw new ProgrammerError("CHECK: " + msg);
    }

    // -----------------------------------------------------
    // ProgrammerError is a Runtime Exception, meaning users does
    // not need to declare the exception in the throws statement.
    // It is made private so that nobody tries to catch it. The
    // intension is to stop the excecution and point out where the
    // contract was violated.

    private static class ProgrammerError extends RuntimeException
    {
        protected ProgrammerError() {super();}
        protected ProgrammerError(String s) {super(s);}
    }

}
