package com.amichel;

import java.lang.Math;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Enumeration;

public class QuoteParser implements FieldConstants
{

	abstract class Field
	{
		Value _value;
		int _id;
		String _name;

		Field( String name, int id, Value value )
		{
			_name = name;
			_id = id;
			_value = value;
		}

		Value getValue() { return _value; }
		String getName() { return _name; }
		int getId() { return _id; }
	}

	abstract class LongField extends Field
	{
		public LongField( String name, int id, String value )
		{
			super( name, id, new LongValue( value ) );
		}
	}

	abstract class DoubleField extends Field
	{
		public DoubleField( String name, int id, String value )
		{
			super( name, id, new DoubleValue( value ) );
		}
	}

	class BidSizeField extends LongField
	{
		public BidSizeField( String value )
		{
			super(  "bid_size", BID_SIZE, value );
		}
	}

	class AskSizeField extends LongField
	{
		public AskSizeField( String value )
		{
			super( "ask_size", ASK_SIZE, value );
		}
	}

	class BidField extends DoubleField
	{
		public BidField( String value )
		{
			super(  "bid", BID, value );
		}
	}

	class AskField extends DoubleField
	{
		public AskField( String value )
		{
			super(  "ask", ASK, value );
		}
	}

	class LastField extends DoubleField
	{
		public LastField( String value )
		{
			super(  "last", LAST, value );
		}
	}

	class IdField extends Field
	{
		public IdField( String name, int id, String value )
		{
			super( name, id, new IdValue( value ) );
		}
	}

	class BidIdField extends IdField
	{
		public BidIdField( String value )
		{
			super(  "bid_id", BID_ID, value );
		}
	}

	class AskIdField extends IdField
	{
		public AskIdField( String value )
		{
			super(  "ask_id", ASK_ID, value );
		}
	}

	class VolumeField extends LongField
	{
		public VolumeField( String value )
		{
			super(  "volume", VOLUME, value );
		}
	}

	class TimeField extends Field
	{
		public TimeField( String name, int id, String value )
		{
			super( name, id, new TimeValue( value ) );
		}
	}

	class TradeTimeField extends TimeField
	{
		public TradeTimeField( String value )
		{
			super(  "trade_time", TRADE_TIME, value );
		}
	}

	class QuoteTimeField extends TimeField
	{
		public QuoteTimeField( String value )
		{
			super(  "quote_time", QUOTE_TIME, value );
		}
	}

	class LowField extends DoubleField
	{
		public LowField( String value )
		{
			super(  "low", LOW, value );
		}
	}

	class HighField extends DoubleField
	{
		public HighField( String value )
		{
			super (  "high", HIGH, value );
		}
	}

	class PrevCloseField extends DoubleField
	{
		public PrevCloseField( String value )
		{
			super(  "prev_close", PREV_CLOSE, value );
		}
	}

	class FieldMap
	{
		private Dictionary map = new Hashtable();
		private String _symbol;

		void setSymbol( String symbol ) { _symbol = symbol; }
		String getSymbol() { return _symbol; }

		void addField( int n, Field field )
		{
			map.put( new Integer( n ), field );
		}

		public Enumeration elements()
		{
			return map.elements();
		}
	}

	public int processField( String str, int n, FieldMap fm )
	{
		char c = str.charAt( n );

		switch( c )
		{
		case BID:
				//bid
			fm.addField( c, new BidField( str.substring( ++n, n += 4 ) ) );
			break;
		case ASK:
				//ask
			fm.addField( c, new AskField( str.substring( ++n, n += 4 ) ) );
			break;
		case LAST:
				//last
			fm.addField( c, new LastField( str.substring( ++n, n += 4 ) ) );
			break;
		case BID_SIZE:
				// bid size
			fm.addField( c, new BidSizeField( str.substring( ++n, n += 4 ) ) );
			break;
		case ASK_SIZE:
				// ask size
			fm.addField( c, new AskSizeField( str.substring( ++n, n += 4 ) ) );
			break;
		case BID_ID:
				// bid id
			fm.addField( c, new BidIdField( str.substring( ++n, n += 2 ) ) );
			break;
		case ASK_ID:
				//ask id
			fm.addField( c, new AskIdField( str.substring( ++n, n += 2 ) ) );
			break;
		case VOLUME:
				// volume
			fm.addField( c, new VolumeField( str.substring( ++n, n += 8 ) ) );
			break;
		case LAST_SIZE:
				// last size
//			fm.addField( c, new AskField( str.substring( n + 1, 4 ) ) );
			break;
		case TRADE_TIME:
				//trade time
			fm.addField( c, new TradeTimeField( str.substring( ++n, n += 4 ) ) );
			break;
		case QUOTE_TIME:
				// quote time
			fm.addField( c, new QuoteTimeField( str.substring( ++n, n += 4 ) ) );
			break;
		case HIGH:
				// high
			fm.addField( c, new HighField( str.substring( ++n, n += 4 ) ) );
			break;
		case LOW:
				// low
			fm.addField( c, new LowField( str.substring( ++n, n += 4 ) ) );
			break;
		case BT:
				// BT
			//fm.addField( c, new AskField( str.substring( n + 1, 4 ) ) );
			break;
		case PREV_CLOSE:
				// prev close
			fm.addField( c, new PrevCloseField( str.substring( ++n, n += 4 ) ) );
			break;
		case EXCHANGE:
				// exchange
//			fm.addField( c, new AskField( str.substring( ++n, n += 4 ) ) );
			break;
		case ISLAND_BID:
				// island bid
			break;
		case ISLAND_ASK:
				// island ask
			break;
		case ISLAND_VOLUME:
				// island volume
			break;
		default:
			break;
		}
		return n;
	}

	public FieldMap parseQuote( String str )
	{
		if( str.charAt( 0 ) == 'S' )
		{
			int n = str.charAt( 7 );

			FieldMap fm = new FieldMap();

			fm.setSymbol( str.substring( 8, 8 + n ) );

			n += 8;
			while( ( n = processField( str, n, fm ) ) < str.length() );
			return fm;
		}
		else
			return null;
	}
}

	abstract class Value
	{
	}

	class DoubleValue extends Value
	{
		private double _value;

		DoubleValue( String value )
		{
			_value = makeDouble( value );
		}

		double getValue() { return _value; }
		private double makeDouble( String str )
		{
/*			long c1 = str.charAt( 0 ) & 0xff;
			long l1 = c1 << 24;
			long c2 = str.charAt( 1 ) & 0xff;
			long l2 = c2 << 16;
			long c3 = str.charAt( 2 ) & 0xff;
			long l3 = c3 << 8;
			long c4 = str.charAt( 3 ) & 0xff;
			long l4 = c4;
			

			long l = l1 + l2 + l3 + l4;			
*/
			long l = ( str.charAt( 0 ) << 24 ) + ( str.charAt( 1 ) << 16 ) + ( str.charAt( 2 ) << 8 )+ str.charAt( 3 );
				// sign indicated by bit 0
			int sign = ( ( l & 0x80000000 ) != 0 ) ? -1 : 1;
				// exponent is 2 ^ ( (bits 1-8) - 127 );
/*			long l5 = l & 0x7f800000;
			l5 = l5 >>> 23;
			l5 = l5 - 127;
		
			double exp = Math.pow( 2, l5 );
*/
			double exp = Math.pow( 2 , ( ( l & 0x7f800000 ) >>> 23 ) - 127 );
				//  mantissa is bits 9 - 31 / 0x800000 + 1;
/*			double d1 = l & 0x7fffff;
			double d2 = 0x800000;
			double mant = d1/d2 + 1;
*/
			double mant = ( (double)( l & 0x7fffff ) ) / ( ( double)0x800000 ) + 1;
				// the value is:
			return sign * exp * mant;
		}

	}

	class LongValue extends Value
	{
		private long _value;

		LongValue( String value )
		{
			_value = makeLong( value );
		}

		long getValue() { return _value; }

		private long makeLong( String str )
		{
				// streamer actually allocates 16 bytes, so we will use a double, instead of a long, which is 8 bytes
			double d = 0;

			int l= str.length();
			for( int n = 0; n < l; n++ )
			{
				char c = str.charAt( n );
				double x = (double)c;
				int exp = ( l - 1 - n )*8;
				double y = Math.pow( ( double )2, exp );
				d +=  x * y;
			}

			return (long)d;
		}
	}

	class IdValue extends Value
	{
		IdValue( String value )
		{
		}
	}

	class TimeValue extends Value
	{
		public
			TimeValue( String value )
			{
			}
	}
